% fitgpr.m
% ========
%
% This function requires certain functions from the gpml suite of Matlab
% functions (see http://www.GaussianProcess.org/gpml/code and also the GPML
% book: Gaussian Processes for Machine Learning, Carl Edward Rasmussen and
% Christopher K. I. Williams, The MIT Press, 2006. ISBN 0-262-18253-X).
% These are:
%               o   solve_chol
%               o   sq_dist
%               o   gpr
%               o   covNoise
%               o   covSEiso
%               o   covSum
%               o   minimize
%
% To Run:
% =======
% fitgpr() or fitgpr          
%     -  Runs the demo (including online figure creation)
% storedsamples = fitgpr
%     -  Runs the demo, returning bootstrap samples in "storedsamples"
% storedsamples = fitgpr(data, numberOfSamples, drawFigures)
%     -  Performs GPR bootstrapping on the data sample "data", returning
%        "numberOfSamples" bootstrap samples, and optionally drawing 
%        figures.  
%        "data" - should be of the form [x;y] (defaults to demo data)
%        "numberOfSamples" - number of bootstrap samples (defaults to 200)
%        "drawFigures" - 1 (default) or 0.  If 0, figures suppressed. 


function storedsamples = fitgpr(data, numberOfSamples, drawFigures)

% Housekeeping and Initialization:
% ================================

if nargin<1,
    % If no data set provided, make a demonstration one to use:
    input  = [0;2;4;6;8;10;12;14;16;18;20;25;30;40;50;60];
    output = [0;0.3315;0.8645;0.9635;0.9279;0.8162;0.7553;0.7680;...
        0.8416;0.7680;0.8010;0.7832;0.8086;0.4888;0.2782;0.2553];
else
    % Otherwise, use the data set provided:
    input  = data(:,1);
    output = data(:,2);
end

if nargin<2,
    numberOfSamples = 200;
end

if nargin<3,
    drawFigures = 1;
end


% Specify the covariance function for the GP regressor:
covfunc = {'covSum', {'covSEiso','covNoise'}};


% Estimate the ML hyperparameters:
% ================================
% Initial starting point for optimization routine:
loghyper = [log(1.0); log(1.0); log(0.1)];

% Now optimize the hyperparameters
loghyper = minimize(loghyper, 'gpr', -500, covfunc, input, output);



% Plot the resulting GP regressor:
% ================================

if (drawFigures)
    % Gridpoints on which to evaluate the GP regressor:
    inputtest = linspace(0,max(input), 500)';
    % Make predictions at the gridpoints (including confidence intervals):
    [mu S2] = gpr(loghyper, covfunc, input, output, inputtest);

    figure;
    clf % Clear figure (should not be necessary)

    % Create the polygon for the fill object:
    f = [mu+3*sqrt(S2);flipdim(mu-3*sqrt(S2),1)];
    % Fill
    fill([inputtest; flipdim(inputtest,1)], f, [7 7 7]/8, 'EdgeColor',...
        [7 7 7]/8);
    hold on % Allow further plot objects to be superimposed on this figure
    % Plot the mean (i.e. prediction):
    plot(inputtest,mu,'r','LineWidth',2);
    % Plot the training data as well:
    plot(input, output, 'bo', 'MarkerFaceColor', [0.78 0.702 1]);
    xlabel('Time, t')
    ylabel('Output data point, y')
    disp(['We will now draw ', num2str(numberOfSamples),...
        ' bootstrap samples from the GP posterior.'])
    disp('Press any key to continue.')
    pause
    saveinputtest = inputtest;
end


% Draw (and plot) bootstrap samples
% ================================

% Initialize some variables required for sampling from the multivariate
% Gaussian:

[muinput S2input] = gpr(loghyper, covfunc, input, output, input);

inputtest = input;

K = feval(covfunc{:}, loghyper, input);
[temp, Kstar] = feval(covfunc{:}, loghyper, input, inputtest);
Kss = feval(covfunc{:}, loghyper, inputtest);

aaa = Kstar'*inv(K)*output;
bbb = Kss - (Kstar'*inv(K)*Kstar);

storedsamples = []; % We will save all of the bootstrap samples drawn
for (i=1:numberOfSamples)
    hold off % We want to plot each bootstrap sample, one at a time
    % Draw a sample from the multivariate Gaussian
    y = aaa+chol(bbb)'*randn(length(inputtest),1);
    storedsamples = [storedsamples, y];
    if (drawFigures)
        fill([saveinputtest; flipdim(saveinputtest,1)],...
            f, [7 7 7]/8, 'EdgeColor', [7 7 7]/8);
        hold on
        plot(saveinputtest,mu,'r','LineWidth',2);
        plot(input, output, 'bo', 'MarkerFaceColor', [0.78 0.702 1]);
        plot(inputtest,  y, 'ro', 'MarkerFaceColor', [1 0.78 0.702]);
        xlabel('Time, t');
        ylabel('Output data point, y');
        title(['Sample ', num2str(i)]);
        legend('+/- 3 s.d. intervals','Mean prediction',...
            'Observed data','Bootstrap Sample');
        pause(0.1);% Give the user time to see each bootstrap sample
    end
end

if(drawFigures)
    % Plot the marginals
    % ================================
    disp('We now plot the marginals at each of the observed data points.')
    disp('Press any key to continue.')
    pause

    % Having plotted each of the individual samples, now create a figure of ALL
    % of the samples, and the marginal distributions:
    figure; hold on; % New figure

    % Initially, consider just the first marginal (this is a Matlab trick to
    % make specifying the plot legend easier)

    % Specify the mean and standard deviation for the first marginal:
    mu = muinput(1); sigma = sqrt(S2input(1));
    % Specify the "width" of the first marginal:
    xvec = linspace(mu - 3*sigma, mu+ 3*sigma, 1000);
    % Calculate the shape of the first marginal (i.e. Normal centred at mu, and
    % with standard deviation given by sigma:
    yvec = (1/(sigma*sqrt(2*pi)))*exp(-((xvec - mu).^2)./(2*sigma*sigma));
    h3 = fill(yvec+input(1), xvec, [7 7 7]/8, 'EdgeColor', [7 7 7]/8);

    % Now plot all of the remaining marginals:
    for(i = 2:length(muinput))
        % Specify the means and standard deviations for each of the marginals:
        mu = muinput(i); sigma = sqrt(S2input(i));
        % Specify the "width" of each marginal:
        xvec = linspace(mu - 3*sigma, mu+ 3*sigma, 1000);
        % Calculate the shape of each marginal (i.e. Normal centred at mu, with
        % standard deviation given by sigma:
        yvec = (1/(sigma*sqrt(2*pi)))*exp(-((xvec - mu).^2)./(2*sigma*sigma));
        fill(yvec+input(i), xvec, [7 7 7]/8, 'EdgeColor', [7 7 7]/8);
    end

    % Plot the bootstrap samples:
    h1 = plot(inputtest, storedsamples(:,1), 'rx');
    plot(inputtest, storedsamples, 'rx');
    % And also the original data:
    h2 = plot(input, output, 'bo', 'MarkerFaceColor', [0.78 0.702 1]);
    % Finally, label:
    xlabel('Time, t')
    ylabel('Output data point, y')

    legend([h1 h2 h3], {'Bootstrap samples'; 'Observed data'; 'Marginal distributions'})
end